'' ------------------------------------------------------------------------ ''
''  @@ Source Documentation                      *** Basic Version ***      ''
''                                                                          ''
''  TITLE : DEMOMIX.BAS                                                     ''
''                                                                          ''
''  DESCRIPTION :                                                           ''
''      This program demostrates how to use the AUXDRV.DRV driver to        ''
''      get and set the mixer setting.                                      ''
''      file.                                                               ''
''                                                                          ''
''      Note that the BLASTER environment has to be set before executing    ''
''      this program.                                                       ''
''                                                                          ''
''  Note :                                                                  ''
''      Use switch /Fs for Microsoft Basic PDS 7.1 compiler.                ''
''                                                                          ''
''  Copyright (c) Creative Technology Ltd, 1993. All rights reserved.       ''
''                                                                          ''
'' ------------------------------------------------------------------------ ''

'$INCLUDE: 'SBKAUX.BI'
'$INCLUDE: 'SBKX.BI'

CONST   DRIVERxSIZE = 32768&

TYPE structMIXER
    inpSwitch   AS LONG     ' Input source switches
    outSwitch   AS LONG     ' Output source switches
    stAGC       AS INTEGER  ' AGC status - on oR off
    inpGain     AS INTEGER  ' Input gain - Range (0 - 3)
    outGain     AS INTEGER  ' Output gain - Range (0 - 3)
    Bass        AS INTEGER  ' Bass level - Range (0 - 255)
    Treble      AS INTEGER  ' Treble level - Range (0 - 255)
    volMaster   AS INTEGER  ' Master volume level - Range (0 - 255)
    volVoice    AS INTEGER  ' Voice volume level - Range (0 -255)
    volMidi     AS INTEGER  ' Midi volume level - Range (0 -255)
    volCD       AS INTEGER  ' CD volume level - Range (0 -255)
    volLine     AS INTEGER  ' Line in volume level - Range (0 -255)
    volMic      AS INTEGER  ' Mic volume level - Range (0 -255)
    volPcSpker  AS INTEGER  ' PC speaker volume level - Range (0 -255)
END TYPE

DECLARE FUNCTION PrepareCTAUXDrv%(BlasterAddx AS LONG)
DECLARE SUB GetMixer(ptrMixer AS structMIXER)
DECLARE SUB DisplayMixer(ptrMixer AS structMIXER)
DECLARE SUB SetMixer(ptrMixer AS structMIXER)
DECLARE SUB PrepareMixSet(ptrMixer AS structMIXER)

REM $DYNAMIC
CLEAR

' Set memory for AUXDRV.DRV
Dummy = SETMEM(-DRIVERxSIZE)

    DIM  prevSetting AS structMIXER
    DIM  currSetting AS structMIXER
    DIM BlasterEnv AS STRING
    DIM BlasterAddx AS LONG

    ' Retrieve the BLASTER environment settings
    BlasterEnv = ENVIRON$("BLASTER")
    IF LEN(BlasterEnv) <> 0 THEN

        BlasterAddx = sbkMakeAsciizString&(BlasterEnv)

        ' Loads AUXDRV.DRV into memory
        IF PrepareCTAUXdrv%(BlasterAddx) <> 0 THEN

            ' Initialises AUXDRV.DRV
            CALL ctadInit

            ' Stores mixer setting
            PRINT "Saving current mixer settings..."
            CALL GetMixer(prevSetting)

            ' Displays mixer setting
            INPUT "Enter to display current mixer settings...";in%
            CALL DisplayMixer(prevSetting)

            ' Fills in the mixer structure
            CALL PrepareMixSet(currSetting)

            ' Sets mixer
            PRINT "Changing the mixer settings..."
            CALL SetMixer(currSetting)
            INPUT "Enter to view the changes on mixer settings...";in%

            ' Displays mixer setting
            CALL DisplayMixer(currSetting)

            ' Restores previous mixer setting
            PRINT "Restoring back the previous mixer settings..."
            CALL SetMixer(prevSetting)
            INPUT "Enter to display current mixer settings...";in%

            ' Display mixer settings
            CALL DisplayMixer(prevSetting)

            ' Terminates AUXDRV.DRV
            CALL ctadTerminate

            PRINT "End..."
        END IF
    ELSE
        PRINT "BLASTER environment not set."
    END IF

'Return memory to system
Dummy = SETMEM(DRIVERxSIZE)

END


'-------------------------------------------------------------------------- '
'   @@ Usage                                                                '
'                                                                           '
'   FUNCTION PrepareCTAUXDrv%(BlasterAddx AS LONG)                          '
'                                                                           '
'   Description :                                                           '
'       Load and endorses AUXDRV.DRV.                                       '
'                                                                           '
'   Entry :                                                                 '
'       BlasterAddx - Far address of BLASTER setting.                       '
'                                                                           '
'   Exit :                                                                  '
'       segment of the driver if sucessful else return 0                    '
'                                                                           '
'---------------------------------------------------------------------------'

FUNCTION PrepareCTAUXDrv%(BlasterAddx AS LONG)

    CTAuxDrv% = sbkLoadDriver%("AUXDRV.DRV",UNUSED)
    IF CTAuxDrv% <> 0 THEN

        ' Initialises the CTAUX.DRV driver entry point
        CALL ctadSetDriverEntry((CTAuxDrv%))

        ' Retrieves AUXDRV.DRV version
        IF ctadGetDrvVer% >= &H0302 THEN

            ' Passes BLASTER environment settings to the driver
            IF ctadGetEnvSettings%((BlasterAddx)) = 0 THEN
                PrepareCTAUXDrv% = CTAuxDrv%
                EXIT FUNCTION
            ELSE
                PRINT "BLASTER environment is not valid"
            END IF
        ELSE
            PRINT "Invalid AUXDRV.DRV - ";
            PRINT "I need AUXDRV.DRV version 3.02 or higher."
        END IF

        CALL sbkFreeMem(CTAuxDrv%)
    ELSE
        PRINT "Error loading AUXDRV.DRV or AUXDRV.DRV not found."
    END IF

    PrepareCTAUXDrv% = 0

END FUNCTION


'---------------------------------------------------------------------------'
'   @@ Usage                                                                '
'                                                                           '
'   SUB GetMixer(ptrMixer AS structMIXER)                                              '
'                                                                           '
'   Description :                                                           '
'       Retrieves and stores the mixer setting.                             '
'                                                                           '
'   Entry :                                                                 '
'       ptrMixer - mixer structure where the mixer                          '
'                  setting is to be stored.                                 '
'                                                                           '
'   Exit :                                                                  '
'       none.                                                               '
'                                                                           '
'---------------------------------------------------------------------------'

SUB GetMixer(ptrMixer AS structMIXER)

    ' Gets input sources switch
    ptrMixer.inpSwitch = ctadGetMixerSwitch&(0)
    ' Gets output sources switch
    ptrMixer.outSwitch = ctadGetMixerSwitch&(1)

    ' Gets AGC status
    ptrMixer.stAGC = ctadGetAGC%

    ' Get mixer input gain
    ptrMixer.inpGain = ctadGetMixerGain%(0)
    ' Get mixer output gain
    ptrMixer.outGain = ctadGetMixerGain%(1)

    ' Gets mixer bass level
    ptrMixer.Bass = ctadGetToneLevel%(1)
    ' Gets mixer treble level
    ptrMixer.Treble = ctadGetToneLevel%(0)

    ' Gets master, voice, CD, line-in, mic and PC speaker volume level
    ptrMixer.volMaster = ctadGetVolume%(MIXERVOLxMASTER)
    ptrMixer.volVoice = ctadGetVolume%(MIXERVOLxVOICE)
    ptrMixer.volMidi = ctadGetVolume%(MIXERVOLxMIDI)
    ptrMixer.volCD = ctadGetVolume%(MIXERVOLxCD)
    ptrMixer.volLine = ctadGetVolume%(MIXERVOLxLINE)
    ptrMixer.volMic = ctadGetVolume%(MIXERVOLxMIC)
    ptrMixer.volPcSpker = ctadGetVolume%(MIXERVOLxPCSPEAKER)

END SUB


' --------------------------------------------------------------------------'
'   @@ Usage                                                                '
'                                                                           '
'   SUB SetMixer(structMIXER ptrMixer)                                      '
'                                                                           '
'   Description :                                                           '
'       Sets mixer.                                                         '
'                                                                           '
'   Entry :                                                                 '
'       ptrMixer - mixer structure which specifies                          '
'                  the mixer setting.                                       '
'                                                                           '
'   Exit :                                                                  '
'       none.                                                               '
' --------------------------------------------------------------------------'

SUB SetMixer(ptrMixer AS structMIXER)

    DIM retVal AS INTEGER


    ' Sets input sources switch
    retVal = ctadSetMixerSwitch% (0,ptrMixer.inpSwitch)
    ' Sets output sources switch
    retVal = ctadSetMixerSwitch%(1,ptrMixer.outSwitch)

    ' Sets AGC status
    retVal = ctadSetAGC%(ptrMixer.stAGC)

    ' Set mixer input gain
    retVal = ctadSetMixerGain%(0,ptrMixer.inpGain)
    ' Set mixer output gain
    retVal = ctadSetMixerGain%(1,ptrMixer.outGain)

    ' Sets mixer bass level
    retVal = ctadSetToneLevel%(1,ptrMixer.Bass)
    ' Sets mixer treble level
    retVal = ctadSetToneLevel%(0,ptrMixer.Treble)

    ' Sets master, voice, CD, line-in, mic and PC speaker volume level
    retVal = ctadSetVolume%(MIXERVOLxMASTER,ptrMixer.volMaster)
    retVal = ctadSetVolume%(MIXERVOLxVOICE,ptrMixer.volVoice)
    retVal = ctadSetVolume%(MIXERVOLxMIDI,ptrMixer.volMidi)
    retVal = ctadSetVolume%(MIXERVOLxCD,ptrMixer.volCD)
    retVal = ctadSetVolume%(MIXERVOLxLINE,ptrMixer.volLine)
    retVal = ctadSetVolume%(MIXERVOLxMIC,ptrMixer.volMic)
    retVal = ctadSetVolume%(MIXERVOLxPCSPEAKER,ptrMixer.volPcSpker)

END SUB


'---------------------------------------------------------------------------'
'   @@ Usage                                                                '
'                                                                           '
'   SUB PrepareMixerSetting(ptrMixer AS structMIXER)                        '
'                                                                           '
'   Description :                                                           '
'       Fills in the mixer structure correspondingly.                       '
'                                                                           '
'   Entry :                                                                 '
'       ptrMixer - mixer structure to be set.                               '
'                                                                           '
'   Exit :                                                                  '
'       none.                                                               '
'                                                                           '
'---------------------------------------------------------------------------'

SUB PrepareMixSet(ptrMixer AS structMIXER)

    DIM  dwValue AS LONG
    DIM  wOff AS INTEGER

    ' initialises to zero
    dwValue = LEN(ptrMixer)
    wOff = VARPTR(ptrMixer)
    DEF SEG = VARSEG(ptrMixer)
    FOR I%=0 TO dwValue-1
        POKE wOff+I%, 0
    NEXT I%
    DEF SEG

    ' sets MIDI as stereo input source
    ptrMixer.inpSwitch = sbkMakeDWord&(MIXERSWIxMIDIxL,MIXERSWIxMIDIxR)

    ' sets both the left and right input gain to 2
    ptrMixer.inpGain = sbkMakeWord%(2,2)

    ' sets CD as ouput source
    ptrMixer.outSwitch = MIXERSWIxCDxR + MIXERSWIxCDxL

    ' sets both the left and right output gain to 2
    ptrMixer.outGain = sbkMakeWord%(2,2)

    ' sets both the right and left treble level to maximum
    ptrMixer.Treble = &Hffff
    ' sets both the right and left bass level to maximum
    ptrMixer.Bass = &Hffff

    ' sets both the right and left Master volume to 3/4
    ' of the maximum level
    wOff = &Hff * 3 / 4
    ptrMixer.volMaster = sbkMakeWord%(wOff,wOff)

    ' sets both the right and left CD volume to 3/4
    ' of the maximum level
    ptrMixer.volCD = sbkMakeWord%(wOff,wOff)

END SUB


'---------------------------------------------------------------------------'
'  @@ Usage                                                                 '
'                                                                           '
'  SUB DisplayMixer(ptrMixer AS structMIXER)                                '
'                                                                           '
'  Description :                                                            '
'      Displays mixer setting.                                              '
'                                                                           '
'  Entry :                                                                  '
'      ptrMixer - mixer structure which the mixer                           '
'                 setting is to be displayed.                               '
'                                                                           '
'  Exit :                                                                   '
'      none.                                                                '
'                                                                           '
'---------------------------------------------------------------------------'

SUB DisplayMixer(ptrMixer AS structMIXER)

    DIM OnOrOff$(0 TO 1,0 TO 3)


    OnOrOff$(0,0) = "OFF"
    OnOrOff$(1,0) = "ON "

    PRINT TAB(12);"-------------- Mixer Settings ---------------"

    PRINT "Input sources",,"Left Channel",,"Right Channel"

    PRINT TAB(4);"Mic             :",
    PRINT OnOrOff$((sbkHighWord%(ptrMixer.inpSwitch) AND MIXERSWIxMIC)_
                   / MIXERSWIxMIC,0),,
    PRINT OnOrOff$((ptrMixer.inpSwitch AND MIXERSWIxMIC)_
                   / MIXERSWIxMIC,0)

    PRINT TAB(4);"CD              :",
    PRINT "L:";OnOrOff$((sbkHighWord%(ptrMixer.inpSwitch) AND MIXERSWIxCDxL)_
                        / MIXERSWIxCDxL,0);
    PRINT "  R:";OnOrOff$((sbkHighWord%(ptrMixer.inpSwitch) AND MIXERSWIxCDxR)_
                        / MIXERSWIxCDxR,0),,
    PRINT "L:";OnOrOff$((ptrMixer.inpSwitch AND MIXERSWIxCDxL)_
                        / MIXERSWIxCDxL,0);
    PRINT "  R:";OnOrOff$((ptrMixer.inpSwitch AND MIXERSWIxCDxR)_
                        / MIXERSWIxCDxR,0)

    PRINT TAB(4);"LINE            :",
    PRINT "L:";OnOrOff$((sbkHighWord%(ptrMixer.inpSwitch) AND MIXERSWIxLINExL)_
                        / MIXERSWIxLINExL,0);
    PRINT "  R:";OnOrOff$((sbkHighWord%(ptrMixer.inpSwitch) AND MIXERSWIxLINExR)_
                        / MIXERSWIxLINExR,0),,
    PRINT "L:";OnOrOff$((ptrMixer.inpSwitch AND MIXERSWIxLINExL)_
                        / MIXERSWIxLINExL,0);
    PRINT "  R:";OnOrOff$((ptrMixer.inpSwitch AND MIXERSWIxLINExR)_
                        / MIXERSWIxLINExR,0)

    PRINT TAB(4);"MIDI            :",
    PRINT "L:";OnOrOff$((sbkHighWord%(ptrMixer.inpSwitch) AND MIXERSWIxMIDIxL)_
                        / MIXERSWIxMIDIxL,0);
    PRINT "  R:";OnOrOff$((sbkHighWord%(ptrMixer.inpSwitch) AND MIXERSWIxMIDIxR)_
                        / MIXERSWIxMIDIxR,0),,
    PRINT "L:";OnOrOff$((ptrMixer.inpSwitch AND MIXERSWIxMIDIxL)_
                        / MIXERSWIxMIDIxL,0);
    PRINT "  R:";OnOrOff$((ptrMixer.inpSwitch AND MIXERSWIxMIDIxR)_
                        / MIXERSWIxMIDIxR,0)

    PRINT"Output sources"

    PRINT TAB(4);"Mic             :",
    PRINT OnOrOff$((ptrMixer.outSwitch AND MIXERSWIxMIC)_
                        / MIXERSWIxMIC,0)

    PRINT TAB(4);"CD              :",
    PRINT "L:";OnOrOff$((ptrMixer.outSwitch AND MIXERSWIxCDxL)_
                        / MIXERSWIxCDxL,0);
    PRINT "  R:";OnOrOff$((ptrMixer.outSwitch AND MIXERSWIxCDxR)_
                        / MIXERSWIxCDxR,0)

    PRINT TAB(4);"Line            :",
    PRINT "L:";OnOrOff$((ptrMixer.outSwitch AND MIXERSWIxLINExL)_
                        / MIXERSWIxLINExL,0);
    PRINT "  R:";OnOrOff$((ptrMixer.outSwitch AND MIXERSWIxLINExR)_
                        / MIXERSWIxLINExR,0)

    PRINT "AGC status         :",
    PRINT OnOrOff$(ptrMixer.stAGC,0)

    PRINT "Input gain         :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.inpGain);
    PRINT USING "  R:###";sbkHighByte%(ptrMixer.inpGain)

    PRINT "Output gain        :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.outGain);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.outGain)

    PRINT "Bass level         :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.Bass);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.Bass)

    PRINT "Treble level       :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.Treble);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.Treble)

    PRINT "Master volume      :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.volMaster);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.volMaster)

    PRINT "Voice volume       :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.volVoice);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.volVoice)

    PRINT "Midi volume        :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.volMidi);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.volMidi)

    PRINT "CD volume          :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.volCD);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.volCD)

    PRINT "Line volume        :",
    PRINT USING "L:###";sbkHighByte%(ptrMixer.volLine);
    PRINT USING "  R:###";sbkLowByte%(ptrMixer.volLine)

    PRINT "Mic volume         :",
    PRINT USING "###";sbkLowByte%(ptrMixer.volMic)

    PRINT "PC speaker volume  :",
    PRINT USING "###";sbkHighByte%(ptrMixer.volPcSpker)

END SUB
' End Of File
